/**
 * WordPress dependencies
 */
import { createElement } from '@wordpress/element'
import {
	MenuGroup,
	MenuItem,
	ToolbarDropdownMenu,
	VisuallyHidden,
} from '@wordpress/components'
import { useState } from '@wordpress/element'
import { __, _x } from '@wordpress/i18n'
import { external } from '@wordpress/icons'
import { displayShortcut } from '@wordpress/keycodes'
import { useShortcut } from '@wordpress/keyboard-shortcuts'
import MoreMenuDropdown from './more-menu-dropdown'
import MoreMenuFeatureToggle from './more-menu-feature-toggle'

/**
 * Internal dependencies
 */
import KeyboardShortcutHelpModal from '../keyboard-shortcut-help-modal'

const POPOVER_PROPS = {
	className: 'customize-widgets-more-menu__content',
}

export default function MoreMenu() {
	const [
		isKeyboardShortcutsModalActive,
		setIsKeyboardShortcutsModalVisible,
	] = useState(false)
	const toggleKeyboardShortcutsModal = () =>
		setIsKeyboardShortcutsModalVisible(!isKeyboardShortcutsModalActive)

	useShortcut(
		'core/customize-widgets/keyboard-shortcuts',
		toggleKeyboardShortcutsModal
	)

	return (
		<>
			<MoreMenuDropdown
				as={ToolbarDropdownMenu}
				className="customize-widgets-more-menu"
				popoverProps={POPOVER_PROPS}>
				{() => (
					<>
						<MenuGroup label={_x('View', 'noun')}>
							<MoreMenuFeatureToggle
								scope="core/customize-widgets"
								feature="fixedToolbar"
								label={__('Top toolbar')}
								info={__(
									'Access all block and document tools in a single place'
								)}
								messageActivated={__('Top toolbar activated')}
								messageDeactivated={__(
									'Top toolbar deactivated'
								)}
							/>
						</MenuGroup>
						<MenuGroup label={__('Tools')}>
							<MenuItem
								onClick={() => {
									setIsKeyboardShortcutsModalVisible(true)
								}}
								shortcut={displayShortcut.access('h')}>
								{__('Keyboard shortcuts')}
							</MenuItem>
							<MoreMenuFeatureToggle
								scope="core/customize-widgets"
								feature="welcomeGuide"
								label={__('Welcome Guide')}
							/>
							<MenuItem
								role="menuitem"
								icon={external}
								href={__(
									'https://wordpress.org/support/article/block-based-widgets-editor/'
								)}
								target="_blank"
								rel="noopener noreferrer">
								{__('Help')}
								<VisuallyHidden as="span">
									{
										/* translators: accessibility text */
										__('(opens in a new tab)')
									}
								</VisuallyHidden>
							</MenuItem>
						</MenuGroup>
						<MenuGroup label={__('Preferences')}>
							<MoreMenuFeatureToggle
								scope="core/customize-widgets"
								feature="keepCaretInsideBlock"
								label={__('Contain text cursor inside block')}
								info={__(
									'Aids screen readers by stopping text caret from leaving blocks.'
								)}
								messageActivated={__(
									'Contain text cursor inside block activated'
								)}
								messageDeactivated={__(
									'Contain text cursor inside block deactivated'
								)}
							/>
						</MenuGroup>
					</>
				)}
			</MoreMenuDropdown>
			<KeyboardShortcutHelpModal
				isModalActive={isKeyboardShortcutsModalActive}
				toggleModal={toggleKeyboardShortcutsModal}
			/>
		</>
	)
}
